/* Copyright 2023 Dual Tachyon
 * https://github.com/DualTachyon
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 *     Unless required by applicable law or agreed to in writing, software
 *     distributed under the License is distributed on an "AS IS" BASIS,
 *     WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *     See the License for the specific language governing permissions and
 *     limitations under the License.
 */

#include "bk4819.h"
#include "../bsp/dp32g030/gpio.h"
#include "../bsp/dp32g030/portcon.h"
#include "../driver/gpio.h"
#include "../driver/system.h"
#include "../driver/systick.h"
#include "../driver/uart.h"
#include "../misc.h"
#include "settings.h"

/*static const uint16_t FSK_RogerTable[7] = {
    0xF1A2, 0x7446, 0x61A4, 0x6544, 0x4E8A, 0xE044, 0xEA84,
};*/
//uint16_t TONE2_FREQ;

static uint16_t gBK4819_GpioOutState;

bool gRxIdleMode;

const uint8_t DTMF_COEFFS[] = {111, 107, 103, 98, 80,  71,  58,  44,
                               65,  55,  37,  23, 228, 203, 181, 159};

void BK4819_Init(void) {
  GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCN);
  GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCL);
  GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SDA);

  BK4819_WriteRegister(BK4819_REG_00, 0x8000);
  BK4819_WriteRegister(BK4819_REG_00, 0x0000);
  BK4819_WriteRegister(BK4819_REG_37, 0x1D0F);
  BK4819_WriteRegister(BK4819_REG_36, 0x0022);
  BK4819_SetAGC(0);
  BK4819_WriteRegister(BK4819_REG_19, 0x1041);
  BK4819_WriteRegister(BK4819_REG_7D, 0xE94F);
  BK4819_WriteRegister(BK4819_REG_48, 0xB3A8);

  for (uint8_t i = 0; i < ARRAY_SIZE(DTMF_COEFFS); ++i) {
    BK4819_WriteRegister(0x09, (i << 12) | DTMF_COEFFS[i]);
  }

  BK4819_WriteRegister(BK4819_REG_1F, 0x5454);
  BK4819_WriteRegister(BK4819_REG_3E, 0xA037);
  gBK4819_GpioOutState = 0x9000;
  BK4819_WriteRegister(BK4819_REG_33, 0x9000);
  BK4819_WriteRegister(BK4819_REG_3F, 0);
}

static uint16_t BK4819_ReadU16(void) {
  uint8_t i;
  uint16_t Value;

  PORTCON_PORTC_IE = (PORTCON_PORTC_IE & ~PORTCON_PORTC_IE_C2_MASK) |
                     PORTCON_PORTC_IE_C2_BITS_ENABLE;
  GPIOC->DIR = (GPIOC->DIR & ~GPIO_DIR_2_MASK) | GPIO_DIR_2_BITS_INPUT;
  SYSTICK_DelayUs(1);

  Value = 0;
  for (i = 0; i < 16; i++) {
    Value <<= 1;
    Value |= GPIO_CheckBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SDA);
    GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCL);
    SYSTICK_DelayUs(1);
    GPIO_ClearBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCL);
    SYSTICK_DelayUs(1);
  }
  PORTCON_PORTC_IE = (PORTCON_PORTC_IE & ~PORTCON_PORTC_IE_C2_MASK) |
                     PORTCON_PORTC_IE_C2_BITS_DISABLE;
  GPIOC->DIR = (GPIOC->DIR & ~GPIO_DIR_2_MASK) | GPIO_DIR_2_BITS_OUTPUT;

  return Value;
}

uint16_t BK4819_ReadRegister(BK4819_REGISTER_t Register) {
  uint16_t Value;

  GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCN);
  GPIO_ClearBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCL);
  SYSTICK_DelayUs(1);
  GPIO_ClearBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCN);

  BK4819_WriteU8(Register | 0x80);

  Value = BK4819_ReadU16();

  GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCN);
  SYSTICK_DelayUs(1);
  GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCL);
  GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SDA);

  return Value;
}

void BK4819_WriteRegister(BK4819_REGISTER_t Register, uint16_t Data) {
  GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCN);
  GPIO_ClearBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCL);
  SYSTICK_DelayUs(1);
  GPIO_ClearBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCN);
  BK4819_WriteU8(Register);
  SYSTICK_DelayUs(1);
  BK4819_WriteU16(Data);
  SYSTICK_DelayUs(1);
  GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCN);
  SYSTICK_DelayUs(1);
  GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCL);
  GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SDA);
}

void BK4819_WriteU8(uint8_t Data) {
  uint8_t i;

  GPIO_ClearBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCL);
  for (i = 0; i < 8; i++) {
    if ((Data & 0x80U) == 0) {
      GPIO_ClearBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SDA);
    } else {
      GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SDA);
    }
    SYSTICK_DelayUs(1);
    GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCL);
    SYSTICK_DelayUs(1);
    Data <<= 1;
    GPIO_ClearBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCL);
    SYSTICK_DelayUs(1);
  }
}

void BK4819_WriteU16(uint16_t Data) {
  uint8_t i;

  GPIO_ClearBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCL);
  for (i = 0; i < 16; i++) {
    if ((Data & 0x8000U) == 0U) {
      GPIO_ClearBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SDA);
    } else {
      GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SDA);
    }
    SYSTICK_DelayUs(1);
    GPIO_SetBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCL);
    Data <<= 1;
    SYSTICK_DelayUs(1);
    GPIO_ClearBit(&GPIOC->DATA, GPIOC_PIN_BK4819_SCL);
    SYSTICK_DelayUs(1);
  }
}

void BK4819_SetAGC(uint8_t Value) {
  if (Value == 0) {
    BK4819_WriteRegister(BK4819_REG_13, 0x03BE);
    BK4819_WriteRegister(BK4819_REG_12, 0x037B);
    BK4819_WriteRegister(BK4819_REG_11, 0x027B);
    BK4819_WriteRegister(BK4819_REG_10, 0x007A);
    BK4819_WriteRegister(BK4819_REG_14, 0x0019);
    BK4819_WriteRegister(BK4819_REG_49, 0x2A38);
    BK4819_WriteRegister(BK4819_REG_7B, 0x8420);
  } else if (Value == 1) {
    uint8_t i;

    BK4819_WriteRegister(BK4819_REG_13, 0x03BE);
    BK4819_WriteRegister(BK4819_REG_12, 0x037C);
    BK4819_WriteRegister(BK4819_REG_11, 0x027B);
    BK4819_WriteRegister(BK4819_REG_10, 0x007A);
    BK4819_WriteRegister(BK4819_REG_14, 0x0018);
    BK4819_WriteRegister(BK4819_REG_49, 0x2A38);
    BK4819_WriteRegister(BK4819_REG_7B, 0x318C);
    BK4819_WriteRegister(BK4819_REG_7C, 0x595E);
    BK4819_WriteRegister(BK4819_REG_20, 0x8DEF);
    for (i = 0; i < 8; i++) {
      BK4819_WriteRegister(0x06, (i & 7) << 13 | 0x4A << 7 | 0x36);
    }
    /* for (i = 0; i < 8; i++) {
      BK4819_WriteRegister(BK4819_REG_06, ((i << 13) | 0x2500U) + 0x36U);
    } */
  }
}

//OLD CODE
/*void BK4819_ToggleGpioOut(BK4819_GPIO_PIN_t Pin, bool bSet) {
  if (bSet) {
    gBK4819_GpioOutState |= (0x40U >> Pin);
  } else {
    gBK4819_GpioOutState &= ~(0x40U >> Pin);
  }

  BK4819_WriteRegister(BK4819_REG_33, gBK4819_GpioOutState);
}*/

void BK4819_ToggleGpioOut(BK4819_GPIO_PIN_t Pin, bool bSet) {  
  uint16_t tmp = gBK4819_GpioOutState;//
  
  if (bSet) {
    gBK4819_GpioOutState |= (0x40U >> Pin);
  } else {
    gBK4819_GpioOutState &= ~(0x40U >> Pin);
  }

  if(gBK4819_GpioOutState == tmp) return;//уменьшает перезапись когда значения одинаковые.

  BK4819_WriteRegister(BK4819_REG_33, gBK4819_GpioOutState);
}

void BK4819_SetCDCSSCodeWord(uint32_t CodeWord) {
  // Enable CDCSS
  // Transmit positive CDCSS code
  // CDCSS Mode
  // CDCSS 23bit
  // Enable Auto CDCSS Bw Mode
  // Enable Auto CTCSS Bw Mode
  // CTCSS/CDCSS Tx Gain1 Tuning = 51
  BK4819_WriteRegister(
      BK4819_REG_51,
      0 | BK4819_REG_51_ENABLE_CxCSS | BK4819_REG_51_GPIO6_PIN2_NORMAL |
          BK4819_REG_51_TX_CDCSS_POSITIVE | BK4819_REG_51_MODE_CDCSS |
          BK4819_REG_51_CDCSS_23_BIT | BK4819_REG_51_1050HZ_NO_DETECTION |
          BK4819_REG_51_AUTO_CDCSS_BW_ENABLE |
          BK4819_REG_51_AUTO_CTCSS_BW_ENABLE |
          (51U << BK4819_REG_51_SHIFT_CxCSS_TX_GAIN1));

  // CTC1 Frequency Control Word = 2775
  BK4819_WriteRegister(BK4819_REG_07,
                       0 | BK4819_REG_07_MODE_CTC1 |
                           (2775U << BK4819_REG_07_SHIFT_FREQUENCY));

  // Set the code word
  BK4819_WriteRegister(BK4819_REG_08, 0x0000 | ((CodeWord >> 0) & 0xFFF));
  BK4819_WriteRegister(BK4819_REG_08, 0x8000 | ((CodeWord >> 12) & 0xFFF));
}

void BK4819_SetCTCSSFrequency(uint32_t FreqControlWord) {
  uint16_t Config;

  if (FreqControlWord == 2625) { // Enables 1050Hz detection mode
    // Enable TxCTCSS
    // CTCSS Mode
    // 1050/4 Detect Enable
    // Enable Auto CDCSS Bw Mode
    // Enable Auto CTCSS Bw Mode
    // CTCSS/CDCSS Tx Gain1 Tuning = 74
    Config = 0x944A;
  } else {
    // Enable TxCTCSS
    // CTCSS Mode
    // Enable Auto CDCSS Bw Mode
    // Enable Auto CTCSS Bw Mode
    // CTCSS/CDCSS Tx Gain1 Tuning = 74
    Config = 0x904A;
  }
  BK4819_WriteRegister(BK4819_REG_51, Config);
  // CTC1 Frequency Control Word
  BK4819_WriteRegister(BK4819_REG_07, 0 | BK4819_REG_07_MODE_CTC1 |
                                          ((FreqControlWord * 2065) / 1000)
                                              << BK4819_REG_07_SHIFT_FREQUENCY);
}

void BK4819_Set55HzTailDetection(void) {
  // CTC2 Frequency Control Word = round_nearest(25391 / 55) = 462
  BK4819_WriteRegister(BK4819_REG_07, (1U << 13) | 462);
}

void BK4819_EnableVox(uint16_t VoxEnableThreshold,
                      uint16_t VoxDisableThreshold) {
  // VOX Algorithm
  // if(voxamp>VoxEnableThreshold)       VOX = 1;
  // else if(voxamp<VoxDisableThreshold) (After Delay) VOX = 0;
  uint16_t REG_31_Value;

  REG_31_Value = BK4819_ReadRegister(BK4819_REG_31);
  // 0xA000 is undocumented?
  BK4819_WriteRegister(BK4819_REG_46, 0xA000 | (VoxEnableThreshold & 0x07FF));
  // 0x1800 is undocumented?
  BK4819_WriteRegister(BK4819_REG_79, 0x1800 | (VoxDisableThreshold & 0x07FF));
  // Bottom 12 bits are undocumented, 15:12 vox disable delay *128ms
  BK4819_WriteRegister(BK4819_REG_7A,
                       0x289A); // vox disable delay = 128*5 = 640ms
  // Enable VOX
  BK4819_WriteRegister(BK4819_REG_31, REG_31_Value | 4); // bit 2 - VOX Enable
}

/*const uint16_t listenBWRegValues[4] = {
    0x3028,             // 25
    0x4048,             // 12.5
    0b0000000000011000, // was 0x205C, // 6.25
	0b0001001000101000,	// Satcom
};
//stock 0x3028             // 25
//experimental sat 0x4648

void BK4819_SetFilterBandwidth(BK4819_FilterBandwidth_t Bandwidth) {
if	(gTxVfo->ConfigRX.Frequency > 24299900 && gTxVfo->ConfigRX.Frequency < 27000000 && gTxVfo->CHANNEL_BANDWIDTH == BANDWIDTH_WIDE) {
BK4819_WriteRegister(BK4819_REG_43, listenBWRegValues[BK4819_FILTER_BW_WIDER]);
} else {
  BK4819_WriteRegister(BK4819_REG_43, listenBWRegValues[Bandwidth]);
}
}*/

void BK4819_SetFilterBandwidth(BK4819_FilterBandwidth_t Bandwidth) {
  if (Bandwidth > 9)
    return;

//  Bandwidth = 9 - Bandwidth; // fix for incremental order

  // REG_43
  // <15>    0 ???
  //

  static const uint8_t rf[] = {1, 1, 0, 1, 3, 2, 3, 4, 5, 7};

  // <14:12> 4 RF filter bandwidth
  //         0 = 1.7  KHz
  //         1 = 2.0  KHz
  //         2 = 2.5  KHz
  //         3 = 3.0  KHz *W
  //         4 = 3.75 KHz *N
  //         5 = 4.0  KHz
  //         6 = 4.25 KHz
  //         7 = 4.5  KHz
  // if <5> == 1, RF filter bandwidth * 2

  static const uint8_t wb[] = {1, 1, 0, 0, 2, 2, 2, 3, 4, 6};

  // <11:9>  0 RF filter bandwidth when signal is weak
  //         0 = 1.7  KHz *WN
  //         1 = 2.0  KHz
  //         2 = 2.5  KHz
  //         3 = 3.0  KHz
  //         4 = 3.75 KHz
  //         5 = 4.0  KHz
  //         6 = 4.25 KHz
  //         7 = 4.5  KHz
  // if <5> == 1, RF filter bandwidth * 2

  static const uint8_t af[] = {0, 0, 1, 2, 3, 0, 7, 6, 5, 4};

  // <8:6>   1 AFTxLPF2 filter Band Width
  //         1 = 2.5  KHz (for 12.5k channel space) *N
  //         2 = 2.75 KHz
  //         0 = 3.0  KHz (for 25k   channel space) *W
  //         3 = 3.5  KHz
  //         4 = 4.5  KHz
  //         5 = 4.25 KHz
  //         6 = 4.0  KHz
  //         7 = 3.75 KHz

  static const uint8_t bs[] = {2, 0, 1, 1, 0, 2, 2, 2, 2, 2};

  // <5:4>   0 BW Mode Selection
  //         0 = 12.5k
  //         1 =  6.25k
  //         2 = 25k/20k
  //
  // <3>     1 ???
  //
  // <2>     0 Gain after FM Demodulation
  //         0 = 0dB
  //         1 = 6dB
  //
  // <1:0>   0 ???

  const uint16_t val =
      (0u << 15) |     //  0
      (rf[Bandwidth] << 12) | // *3 RF filter bandwidth
      (wb[Bandwidth] << 9) |  // *0 RF filter bandwidth when signal is weak
      (af[Bandwidth] << 6) |  // *0 AFTxLPF2 filter Band Width
      (bs[Bandwidth] << 4) |  //  2 BW Mode Selection 25K
      (1u << 3) |      //  1
      (0u << 2) |      //  0 Gain after FM Demodulation
      (0u << 0);       //  0
  BK4819_WriteRegister(BK4819_REG_43, val);
}

void BK4819_SetupPowerAmplifier(uint16_t Bias, uint32_t Frequency) {
  uint8_t Gain;

  if (Bias > 255) {
    Bias = 255;
  }
//  if ((gSetting_200TX == false && Frequency < 28000000) || (gSetting_200TX == true && Frequency < 24000000)) {
	if (Frequency < gEeprom.UHFFILTER_FREQ * 500000) {
    // Gain 1 = 1
    // Gain 2 = 0
    Gain = 0x08U;
  } else {
    // Gain 1 = 4
    // Gain 2 = 2
    Gain = 0x22U;
  }
  // Enable PACTLoutput
  BK4819_WriteRegister(BK4819_REG_36, (Bias << 8) | 0x80U | Gain);
}

void BK4819_SetFrequency(uint32_t Frequency) {
  BK4819_WriteRegister(BK4819_REG_38, (Frequency >> 0) & 0xFFFF);
  BK4819_WriteRegister(BK4819_REG_39, (Frequency >> 16) & 0xFFFF);
}

uint32_t BK4819_GetFrequency() {
  return (BK4819_ReadRegister(BK4819_REG_39) << 16) |
         BK4819_ReadRegister(BK4819_REG_38);
}

void BK4819_SetupSquelch(uint8_t SquelchOpenRSSIThresh,
                         uint8_t SquelchCloseRSSIThresh,
                         uint8_t SquelchOpenNoiseThresh,
                         uint8_t SquelchCloseNoiseThresh,
                         uint8_t SquelchCloseGlitchThresh,
                         uint8_t SquelchOpenGlitchThresh) {
/*if (gEeprom.SQUELCH_LEVEL != 0 && gEeprom.sqlcor > 0) {
SquelchOpenRSSIThresh = SquelchOpenRSSIThresh - (gEeprom.sqlcor / 3);
SquelchCloseRSSIThresh = SquelchCloseRSSIThresh + (gEeprom.sqlcor / 3);
SquelchOpenNoiseThresh = SquelchOpenNoiseThresh - (gEeprom.sqlcor / 4);
SquelchCloseNoiseThresh = SquelchCloseNoiseThresh + gEeprom.sqlcor;
SquelchCloseGlitchThresh = 52 + (gEeprom.sqlcor / 10);
SquelchOpenGlitchThresh = 64 + (gEeprom.sqlcor / 10);
//SquelchCloseGlitchThresh = SquelchCloseGlitchThresh + (gEeprom.sqlcor / 4);
//SquelchOpenGlitchThresh = SquelchOpenGlitchThresh + gEeprom.sqlcor;
}*/
  BK4819_WriteRegister(BK4819_REG_70, 0);
  BK4819_WriteRegister(BK4819_REG_4D, 0xA000 | SquelchCloseGlitchThresh);
  // 0x6f = 0110 1111 meaning the default sql delays from the datasheet are used
  // (101 and 111)
  BK4819_WriteRegister(BK4819_REG_4E, // 01 101 11 1 00000000
#ifdef ENABLE_FASTER_CHANNEL_SCAN
                                      // faster (but twitchier)
                       (1u << 14) |     //  1 ???
                           (1u << 11) | // *5  squelch = open  delay .. 0 ~ 7
                           (1u << 9) |  // *3  squelch = close delay .. 0 ~ 3
                           SquelchOpenGlitchThresh); //  0 ~ 255
#else
                                      // original (*)
                       0x6F00 | SquelchOpenGlitchThresh);
#endif
  BK4819_WriteRegister(BK4819_REG_4F,
                       (SquelchCloseNoiseThresh << 8) | SquelchOpenNoiseThresh);
  BK4819_WriteRegister(BK4819_REG_78,
                       (SquelchOpenRSSIThresh << 8) | SquelchCloseRSSIThresh);
  BK4819_SetAF(BK4819_AF_MUTE);
  BK4819_RX_TurnOn();
}

void BK4819_SetAF(BK4819_AF_Type_t AF) {
  // AF Output Inverse Mode = Inverse
  // Undocumented bits 0x2040
  BK4819_WriteRegister(BK4819_REG_47, 0x6040 | (AF << 8));
}

uint16_t BK4819_GetRegValue(RegisterSpec s) {
  return (BK4819_ReadRegister(s.num) >> s.offset) & s.mask;
}

void BK4819_SetRegValue(RegisterSpec s, uint16_t v) {
  uint16_t reg = BK4819_ReadRegister(s.num);
  reg &= ~(s.mask << s.offset);
  BK4819_WriteRegister(s.num, reg | (v << s.offset));
}

void BK4819_SetModulation(ModulationType type) {
  const uint8_t modTypeReg47Values[] = {1, 7, 5, 9, 4};

  BK4819_SetAF(modTypeReg47Values[type]);
  BK4819_SetRegValue(afDacGainRegSpec, 0xF);
  BK4819_WriteRegister(0x3D, type == MOD_USB ? 0 : 0x2AAB);
  BK4819_SetRegValue(afcDisableRegSpec, type != MOD_FM);
}

void BK4819_RX_TurnOn(void) {
  // DSP Voltage Setting = 1
  // ANA LDO = 2.7v
  // VCO LDO = 2.7v
  // RF LDO = 2.7v
  // PLL LDO = 2.7v
  // ANA LDO bypass
  // VCO LDO bypass
  // RF LDO bypass
  // PLL LDO bypass
  // Reserved bit is 1 instead of 0
  // Enable DSP
  // Enable XTAL
  // Enable Band Gap
  BK4819_WriteRegister(BK4819_REG_37, 0x1F0F);

  // Turn off everything
  BK4819_WriteRegister(BK4819_REG_30, 0);

  // Enable VCO Calibration
  // Enable RX Link
  // Enable AF DAC
  // Enable PLL/VCO
  // Disable PA Gain
  // Disable MIC ADC
  // Disable TX DSP
  // Enable RX DSP
  BK4819_WriteRegister(BK4819_REG_30, 0xBFF1);
}

void BK4819_SelectFilter(uint32_t Frequency) {
//  if ((gSetting_200TX == false && Frequency < 28000000) || (gSetting_200TX == true && Frequency < 24000000)) {
	if (Frequency < gEeprom.UHFFILTER_FREQ * 500000) {
    BK4819_ToggleGpioOut(BK4819_GPIO4_PIN32_VHF_LNA, true);
    BK4819_ToggleGpioOut(BK4819_GPIO3_PIN31_UHF_LNA, false);
  } else if (Frequency == 0xFFFFFFFF) {
    BK4819_ToggleGpioOut(BK4819_GPIO4_PIN32_VHF_LNA, false);
    BK4819_ToggleGpioOut(BK4819_GPIO3_PIN31_UHF_LNA, false);
  } else {
    BK4819_ToggleGpioOut(BK4819_GPIO4_PIN32_VHF_LNA, false);
    BK4819_ToggleGpioOut(BK4819_GPIO3_PIN31_UHF_LNA, true);
  }
//  if (Frequency > 62999990 && Frequency < 84000000 && gSetting_500TX == true) {
	if (Frequency > 62999990 && Frequency < 84000000) {
	BK4819_WriteRegister(0x3E, 43515);
//	} else if (gSetting_500TX == true) {
	} else {
	BK4819_WriteRegister(0x3E, 36458);
}
}

void BK4819_DisableScramble(void) {
  uint16_t Value;

  Value = BK4819_ReadRegister(BK4819_REG_31);
  BK4819_WriteRegister(BK4819_REG_31, Value & 0xFFFD);
}

void BK4819_EnableScramble(uint8_t Type) {
  uint16_t Value;

  Value = BK4819_ReadRegister(BK4819_REG_31);
  BK4819_WriteRegister(BK4819_REG_31, Value | 2);
  BK4819_WriteRegister(BK4819_REG_71, (Type * 0x0408) + 0x68DC);
}

void BK4819_DisableVox(void) {
  uint16_t Value;

  Value = BK4819_ReadRegister(BK4819_REG_31);
  BK4819_WriteRegister(BK4819_REG_31, Value & 0xFFFB);
}

void BK4819_DisableDTMF(void) { BK4819_WriteRegister(BK4819_REG_24, 0); }

void BK4819_EnableDTMF(void) {
  BK4819_WriteRegister(BK4819_REG_21, 0x06D8);
  BK4819_WriteRegister(BK4819_REG_24,
                       0 | (1U << BK4819_REG_24_SHIFT_UNKNOWN_15) |
                           (24 << BK4819_REG_24_SHIFT_THRESHOLD) |
                           (1U << BK4819_REG_24_SHIFT_UNKNOWN_6) |
                           BK4819_REG_24_ENABLE | BK4819_REG_24_SELECT_DTMF |
                           (14U << BK4819_REG_24_SHIFT_MAX_SYMBOLS));
}

void BK4819_PlayTone(uint16_t Frequency, bool bTuningGainSwitch) {
  uint16_t ToneConfig;

  BK4819_EnterTxMute();
  BK4819_SetAF(BK4819_AF_BEEP);

  if (bTuningGainSwitch == 0) {
    ToneConfig = 0 | BK4819_REG_70_ENABLE_TONE1 |
                 (96U << BK4819_REG_70_SHIFT_TONE1_TUNING_GAIN);
  } else {
    ToneConfig = 0 | BK4819_REG_70_ENABLE_TONE1 |
                 (28U << BK4819_REG_70_SHIFT_TONE1_TUNING_GAIN);
  }
  BK4819_WriteRegister(BK4819_REG_70, ToneConfig);

  BK4819_WriteRegister(BK4819_REG_30, 0);
  BK4819_WriteRegister(BK4819_REG_30, 0 | BK4819_REG_30_ENABLE_AF_DAC |
                                          BK4819_REG_30_ENABLE_DISC_MODE |
                                          BK4819_REG_30_ENABLE_TX_DSP);

  BK4819_SetToneFrequency(Frequency);
}

void BK4819_EnterTxMute(void) { BK4819_WriteRegister(BK4819_REG_50, 0xBB20); }

void BK4819_ExitTxMute(void) { BK4819_WriteRegister(BK4819_REG_50, 0x3B20); }

void BK4819_Sleep(void) {
  BK4819_WriteRegister(BK4819_REG_30, 0);
  BK4819_WriteRegister(BK4819_REG_37, 0x1D00);
}

void BK4819_TurnsOffTones_TurnsOnRX(void) {
  BK4819_WriteRegister(BK4819_REG_70, 0);
  BK4819_SetAF(BK4819_AF_MUTE);
  BK4819_ExitTxMute();
  BK4819_WriteRegister(BK4819_REG_30, 0);
  BK4819_WriteRegister(
      BK4819_REG_30,
      0 | BK4819_REG_30_ENABLE_VCO_CALIB | BK4819_REG_30_ENABLE_RX_LINK |
          BK4819_REG_30_ENABLE_AF_DAC | BK4819_REG_30_ENABLE_DISC_MODE |
          BK4819_REG_30_ENABLE_PLL_VCO | BK4819_REG_30_ENABLE_RX_DSP);
}

#if defined(ENABLE_AIRCOPY)
void BK4819_SetupAircopy(void) {
  BK4819_WriteRegister(BK4819_REG_70, 0x00E0); // Enable Tone2, tuning gain 48
  BK4819_WriteRegister(BK4819_REG_72, 0x3065); // Tone2 baudrate 1200
  BK4819_WriteRegister(
      BK4819_REG_58,
      0x00C1); // FSK Enable, FSK 1.2K RX Bandwidth, Preamble 0xAA or 0x55, RX
               // Gain 0, RX Mode (FSK1.2K, FSK2.4K Rx and NOAA SAME Rx), TX
               // Mode FSK 1.2K and FSK 2.4K Tx
  BK4819_WriteRegister(
      BK4819_REG_5C, 0x5665); // Enable CRC among other things we don't know yet
  BK4819_WriteRegister(
      BK4819_REG_5D, 0x4700); // FSK Data Length 72 Bytes (0xabcd + 2 byte
                              // length + 64 byte payload + 2 byte CRC + 0xdcba)
}
#endif
void BK4819_ResetFSK(void) {
  BK4819_WriteRegister(BK4819_REG_3F, 0x0000); // Disable interrupts
  BK4819_WriteRegister(BK4819_REG_59,
                       0x0068); // Sync length 4 bytes, 7 byte preamble
  SYSTEM_DelayMs(30);
  BK4819_Idle();
}


void BK4819_Idle(void) { BK4819_WriteRegister(BK4819_REG_30, 0x0000); }

void BK4819_ExitBypass(void) {
  BK4819_SetAF(BK4819_AF_MUTE);
  BK4819_WriteRegister(BK4819_REG_7E, 0x302E);
}

void BK4819_PrepareTransmit(void) {
  BK4819_ExitBypass();
  BK4819_ExitTxMute();
  BK4819_TxOn_Beep();
}

void BK4819_TxOn_Beep(void) {
  BK4819_WriteRegister(BK4819_REG_37, 0x1D0F);
  BK4819_WriteRegister(BK4819_REG_52, 0x028F);
  BK4819_WriteRegister(BK4819_REG_30, 0x0000);
  BK4819_WriteRegister(BK4819_REG_30, 0xC1FE);
}

void BK4819_ExitSubAu(void) { BK4819_WriteRegister(BK4819_REG_51, 0x0000); }

void BK4819_EnableRX(void) {
  if (gRxIdleMode) {
    BK4819_ToggleGpioOut(BK4819_GPIO0_PIN28_RX_ENABLE, true);
    BK4819_RX_TurnOn();
  }
}

void BK4819_EnterDTMF_TX(bool bLocalLoopback) {
  BK4819_EnableDTMF();
  BK4819_EnterTxMute();
  if (bLocalLoopback) {
    BK4819_SetAF(BK4819_AF_BEEP);
  } else {
    BK4819_SetAF(BK4819_AF_MUTE);
  }
  BK4819_WriteRegister(BK4819_REG_70,
                       0 | BK4819_REG_70_MASK_ENABLE_TONE1 |
                           (83 << BK4819_REG_70_SHIFT_TONE1_TUNING_GAIN) |
                           BK4819_REG_70_MASK_ENABLE_TONE2 |
                           (83 << BK4819_REG_70_SHIFT_TONE2_TUNING_GAIN));

  BK4819_EnableTXLink();
}

void BK4819_ExitDTMF_TX(bool bKeep) {
  BK4819_EnterTxMute();
  BK4819_SetAF(BK4819_AF_MUTE);
  BK4819_WriteRegister(BK4819_REG_70, 0x0000);
  BK4819_DisableDTMF();
  BK4819_WriteRegister(BK4819_REG_30, 0xC1FE);
  if (!bKeep) {
    BK4819_ExitTxMute();
  }
}

void BK4819_EnableTXLink(void) {
  BK4819_WriteRegister(
      BK4819_REG_30,
      0 | BK4819_REG_30_ENABLE_VCO_CALIB | BK4819_REG_30_ENABLE_UNKNOWN |
          BK4819_REG_30_DISABLE_RX_LINK | BK4819_REG_30_ENABLE_AF_DAC |
          BK4819_REG_30_ENABLE_DISC_MODE | BK4819_REG_30_ENABLE_PLL_VCO |
          BK4819_REG_30_ENABLE_PA_GAIN | BK4819_REG_30_DISABLE_MIC_ADC |
          BK4819_REG_30_ENABLE_TX_DSP | BK4819_REG_30_DISABLE_RX_DSP);
}

void BK4819_PlayDTMF(char Code) {
  switch (Code) {
  case '0':
    BK4819_WriteRegister(BK4819_REG_71, 0x25F3);
    BK4819_WriteRegister(BK4819_REG_72, 0x35E1);
    break;
  case '1':
    BK4819_WriteRegister(BK4819_REG_71, 0x1C1C);
    BK4819_WriteRegister(BK4819_REG_72, 0x30C2);
    break;
  case '2':
    BK4819_WriteRegister(BK4819_REG_71, 0x1C1C);
    BK4819_WriteRegister(BK4819_REG_72, 0x35E1);
    break;
  case '3':
    BK4819_WriteRegister(BK4819_REG_71, 0x1C1C);
    BK4819_WriteRegister(BK4819_REG_72, 0x3B91);
    break;
  case '4':
    BK4819_WriteRegister(BK4819_REG_71, 0x1F0E);
    BK4819_WriteRegister(BK4819_REG_72, 0x30C2);
    break;
  case '5':
    BK4819_WriteRegister(BK4819_REG_71, 0x1F0E);
    BK4819_WriteRegister(BK4819_REG_72, 0x35E1);
    break;
  case '6':
    BK4819_WriteRegister(BK4819_REG_71, 0x1F0E);
    BK4819_WriteRegister(BK4819_REG_72, 0x3B91);
    break;
  case '7':
    BK4819_WriteRegister(BK4819_REG_71, 0x225C);
    BK4819_WriteRegister(BK4819_REG_72, 0x30C2);
    break;
  case '8':
    BK4819_WriteRegister(BK4819_REG_71, 0x225c);
    BK4819_WriteRegister(BK4819_REG_72, 0x35E1);
    break;
  case '9':
    BK4819_WriteRegister(BK4819_REG_71, 0x225C);
    BK4819_WriteRegister(BK4819_REG_72, 0x3B91);
    break;
  case 'A':
    BK4819_WriteRegister(BK4819_REG_71, 0x1C1C);
    BK4819_WriteRegister(BK4819_REG_72, 0x41DC);
    break;
  case 'B':
    BK4819_WriteRegister(BK4819_REG_71, 0x1F0E);
    BK4819_WriteRegister(BK4819_REG_72, 0x41DC);
    break;
  case 'C':
    BK4819_WriteRegister(BK4819_REG_71, 0x225C);
    BK4819_WriteRegister(BK4819_REG_72, 0x41DC);
    break;
  case 'D':
    BK4819_WriteRegister(BK4819_REG_71, 0x25F3);
    BK4819_WriteRegister(BK4819_REG_72, 0x41DC);
    break;
  case '*':
    BK4819_WriteRegister(BK4819_REG_71, 0x25F3);
    BK4819_WriteRegister(BK4819_REG_72, 0x30C2);
    break;
  case '#':
    BK4819_WriteRegister(BK4819_REG_71, 0x25F3);
    BK4819_WriteRegister(BK4819_REG_72, 0x3B91);
    break;
  }
}

void BK4819_PlayDTMFString(const char *pString, bool bDelayFirst,
                           uint16_t FirstCodePersistTime,
                           uint16_t HashCodePersistTime,
                           uint16_t CodePersistTime,
                           uint16_t CodeInternalTime) {
  uint8_t i;
  uint16_t Delay;

  for (i = 0; pString[i]; i++) {
    BK4819_PlayDTMF(pString[i]);
    BK4819_ExitTxMute();
    if (bDelayFirst && i == 0) {
      Delay = FirstCodePersistTime;
    } else if (pString[i] == '*' || pString[i] == '#') {
      Delay = HashCodePersistTime;
    } else {
      Delay = CodePersistTime;
    }
    SYSTEM_DelayMs(Delay);
    BK4819_EnterTxMute();
    SYSTEM_DelayMs(CodeInternalTime);
  }
}

void BK4819_TransmitTone(bool bLocalLoopback, uint32_t Frequency) {
  BK4819_EnterTxMute();
  BK4819_WriteRegister(BK4819_REG_70,
                       BK4819_REG_70_MASK_ENABLE_TONE1 |
                           (96U << BK4819_REG_70_SHIFT_TONE1_TUNING_GAIN));
  BK4819_SetToneFrequency(Frequency);

  BK4819_SetAF(bLocalLoopback ? BK4819_AF_BEEP : BK4819_AF_MUTE);

  BK4819_EnableTXLink();
  SYSTEM_DelayMs(50);
  BK4819_ExitTxMute();
}

void BK4819_GenTail(bool Tail) {
  switch (Tail) {
  case false: // CTC134
    BK4819_WriteRegister(BK4819_REG_52, 0x828F);
    break;
  case true: // CTC55
    BK4819_WriteRegister(BK4819_REG_07, 0x046f);
    break;
/*  case 1: // CTC120
    BK4819_WriteRegister(BK4819_REG_52, 0xA28F);
    break;
  case 2: // CTC180
    BK4819_WriteRegister(BK4819_REG_52, 0xC28F);
    break;
  case 3: // CTC240
    BK4819_WriteRegister(BK4819_REG_52, 0xE28F);
    break;
  case 4: // CTC55
    BK4819_WriteRegister(BK4819_REG_07, 0x046f);
    break;*/
  }
}

void BK4819_PlayCDCSSTail(void)
{
    BK4819_GenTail(false);     // CTC134
    BK4819_WriteRegister(BK4819_REG_51, 0x804A); // 1 0 0 0 0 0 0 0  0  1001010
}

void BK4819_PlayCTCSSTail(void)
{
/*    #ifdef ENABLE_CTCSS_TAIL_PHASE_SHIFT
        BK4819_GenTail(2);       // 180° phase shift
    #else*/
        BK4819_GenTail(true);       // 55Hz tone freq
//    #endif

    // REG_51
    //
    // <15>  0
    //       1 = Enable TxCTCSS/CDCSS
    //       0 = Disable
    //
    // <14>  0
    //       1 = GPIO0Input for CDCSS
    //       0 = Normal Mode (for BK4819 v3)
    //
    // <13>  0
    //       1 = Transmit negative CDCSS code
    //       0 = Transmit positive CDCSS code
    //
    // <12>  0 CTCSS/CDCSS mode selection
    //       1 = CTCSS
    //       0 = CDCSS
    //
    // <11>  0 CDCSS 24/23bit selection
    //       1 = 24bit
    //       0 = 23bit
    //
    // <10>  0 1050HzDetectionMode
    //       1 = 1050/4 Detect Enable, CTC1 should be set to 1050/4 Hz
    //
    // <9>   0 Auto CDCSS Bw Mode
    //       1 = Disable
    //       0 = Enable
    //
    // <8>   0 Auto CTCSS Bw Mode
    //       0 = Enable
    //       1 = Disable
    //
    // <6:0> 0 CTCSS/CDCSS Tx Gain1 Tuning
    //       0   = min
    //       127 = max

    BK4819_WriteRegister(BK4819_REG_51, 0x904A); // 1 0 0 1 0 0 0 0  0  1001010
}

void BK4819_EnableCDCSS(void) {
  BK4819_GenTail(false); // CTC134
  BK4819_WriteRegister(BK4819_REG_51, 0x804A);
}

void BK4819_EnableCTCSS(void) {
  BK4819_GenTail(true); // CTC55
  BK4819_WriteRegister(BK4819_REG_51, 0x904A);
}

uint16_t BK4819_GetRSSI(void) {
  return BK4819_ReadRegister(BK4819_REG_67) & 0x01FF;
}

bool BK4819_GetFrequencyScanResult(uint32_t *pFrequency) {
  uint16_t High = BK4819_ReadRegister(BK4819_REG_0D);
  bool Finished = (High & 0x8000) == 0;

  if (Finished) {
    uint16_t Low = BK4819_ReadRegister(BK4819_REG_0E);
    *pFrequency = (uint32_t)((High & 0x7FF) << 16) | Low;
  }

  return Finished;
}

BK4819_CssScanResult_t BK4819_GetCxCSSScanResult(uint32_t *pCdcssFreq,
                                                 uint16_t *pCtcssFreq) {
  uint16_t High, Low;

  High = BK4819_ReadRegister(BK4819_REG_69);
  if ((High & 0x8000) == 0) {
    Low = BK4819_ReadRegister(BK4819_REG_6A);
    *pCdcssFreq = ((High & 0xFFF) << 12) | (Low & 0xFFF);
    return BK4819_CSS_RESULT_CDCSS;
  }

  Low = BK4819_ReadRegister(BK4819_REG_68);
  if ((Low & 0x8000) == 0) {
    *pCtcssFreq = (Low & 0x1FFF) * 4843 / 10000;
    return BK4819_CSS_RESULT_CTCSS;
  }

  return BK4819_CSS_RESULT_NOT_FOUND;
}

void BK4819_DisableFrequencyScan(void) {
  BK4819_WriteRegister(BK4819_REG_32, 0x0244);
}

void BK4819_EnableFrequencyScan(void) {
  BK4819_WriteRegister(BK4819_REG_32, 0x0245);
}

void BK4819_SetScanFrequency(uint32_t Frequency) {
  BK4819_SetFrequency(Frequency);
  BK4819_WriteRegister(
      BK4819_REG_51,
      0 | BK4819_REG_51_DISABLE_CxCSS | BK4819_REG_51_GPIO6_PIN2_NORMAL |
          BK4819_REG_51_TX_CDCSS_POSITIVE | BK4819_REG_51_MODE_CDCSS |
          BK4819_REG_51_CDCSS_23_BIT | BK4819_REG_51_1050HZ_NO_DETECTION |
          BK4819_REG_51_AUTO_CDCSS_BW_DISABLE |
          BK4819_REG_51_AUTO_CTCSS_BW_DISABLE);
  BK4819_RX_TurnOn();
}

void BK4819_Disable(void) { BK4819_WriteRegister(BK4819_REG_30, 0); }

void BK4819_StopScan(void) {
  BK4819_DisableFrequencyScan();
  BK4819_Disable();
}

uint8_t BK4819_GetDTMF_5TONE_Code(void) {
  return (BK4819_ReadRegister(BK4819_REG_0B) >> 8) & 0x0F;
}

uint8_t BK4819_GetCDCSSCodeType(void) {
  return (BK4819_ReadRegister(BK4819_REG_0C) >> 14) & 3;
}

uint8_t BK4819_GetCTCType(void) {
  return (BK4819_ReadRegister(BK4819_REG_0C) >> 10) & 3;
}

//#if defined(ENABLE_AIRCOPY)
/*void BK4819_SendFSKData(uint16_t *pData) {
  uint8_t i;
  uint8_t Timeout;

  Timeout = 200;

  SYSTEM_DelayMs(20);

  BK4819_WriteRegister(BK4819_REG_3F, BK4819_REG_3F_FSK_TX_FINISHED);
  BK4819_WriteRegister(BK4819_REG_59, 0x8068);
  BK4819_WriteRegister(BK4819_REG_59, 0x0068);

  for (i = 0; i < 36; i++) {
    BK4819_WriteRegister(BK4819_REG_5F, pData[i]);
  }

  SYSTEM_DelayMs(20);

  BK4819_WriteRegister(BK4819_REG_59, 0x2868);

  while (Timeout) {
    if (BK4819_ReadRegister(BK4819_REG_0C) & 1U) {
      break;
    }
    SYSTEM_DelayMs(5);
    Timeout--;
  }

  BK4819_WriteRegister(BK4819_REG_02, 0);
  SYSTEM_DelayMs(20);
  BK4819_ResetFSK();
}*/

void BK4819_SendFSKData(uint8_t *pData)
{
	unsigned int i;
	uint8_t Timeout = 200;

	SYSTEM_DelayMs(20);

	BK4819_WriteRegister(BK4819_REG_3F, BK4819_REG_3F_FSK_TX_FINISHED);
	BK4819_WriteRegister(BK4819_REG_59, 0x8068);
	BK4819_WriteRegister(BK4819_REG_59, 0x0068);
//	SYSTEM_DelayMs(100);
	for (i = 0; i < 36; i++)
//		BK4819_WriteRegister(BK4819_REG_5F, pData[i]);
		BK4819_WriteRegister(BK4819_REG_5F, (pData[i*2+1] << 8) | pData[i*2]);		// Byte -> Word

	SYSTEM_DelayMs(20);

	BK4819_WriteRegister(BK4819_REG_59, 0x2868);

	while (Timeout-- && (BK4819_ReadRegister(BK4819_REG_0C) & 1u) == 0)
		SYSTEM_DelayMs(5);

	BK4819_WriteRegister(BK4819_REG_02, 0);

	SYSTEM_DelayMs(20);

	BK4819_ResetFSK();
}

/*void BK4819_PrepareFSKReceive(void) {
//  BK4819_ResetFSK();
  BK4819_WriteRegister(BK4819_REG_02, 0);
  BK4819_WriteRegister(BK4819_REG_3F, 0);
//  BK4819_RX_TurnOn();
  BK4819_WriteRegister(BK4819_REG_3F, 0 | BK4819_REG_3F_FSK_RX_FINISHED |
                                          BK4819_REG_3F_FSK_FIFO_ALMOST_FULL);
  // Clear RX FIFO
  // FSK Preamble Length 7 bytes
  // FSK SyncLength Selection
  BK4819_WriteRegister(BK4819_REG_59, 0x4068);
  // Enable FSK Scramble
  // Enable FSK RX
  // FSK Preamble Length 7 bytes
  // FSK SyncLength Selection
  BK4819_WriteRegister(BK4819_REG_59, 0x3068);
}*/
//#endif

void BK4819_PlayRoger(void) {
  BK4819_EnterTxMute();
  BK4819_SetAF(BK4819_AF_MUTE);
  BK4819_WriteRegister(BK4819_REG_70, 0xE000);
  BK4819_EnableTXLink();
  SYSTEM_DelayMs(50);
  BK4819_WriteRegister(BK4819_REG_71, 5500 + (500 * gEeprom.ROGER));
  BK4819_ExitTxMute();
  SYSTEM_DelayMs(80);
  BK4819_EnterTxMute();
  BK4819_WriteRegister(BK4819_REG_71, 9500 - (500 * gEeprom.ROGER));
  BK4819_ExitTxMute();
  SYSTEM_DelayMs(80);
  BK4819_EnterTxMute();
  BK4819_WriteRegister(BK4819_REG_70, 0x0000);
  BK4819_WriteRegister(BK4819_REG_30, 0xC1FE);
}

void BK4819_FskClearFifo(void){
	const uint16_t fsk_reg59 = BK4819_ReadRegister(BK4819_REG_59);
	BK4819_WriteRegister(BK4819_REG_59, (1u << 15) | (1u << 14) | fsk_reg59);
}

void BK4819_FskEnableRx(void){
	const uint16_t fsk_reg59 = BK4819_ReadRegister(BK4819_REG_59);
	BK4819_WriteRegister(BK4819_REG_59, (1u << 12) | fsk_reg59);
}

// -----------------------------------------------------------------------------
void init_fsk0(void){
	BK4819_WriteRegister(BK4819_REG_70, 0x00E0); // Enable Tone2, tuning gain 48
	BK4819_WriteRegister(BK4819_REG_58, 0x00C1); // FSK Enable, FSK 1.2K RX Bandwidth, Preamble 0xAA or 0x55, RX Gain 0, RX Mode
 	BK4819_WriteRegister(BK4819_REG_72, 0x3065); // Tone2 baudrate 1200
                   							 // (FSK1.2K, FSK2.4K Rx and NOAA SAME Rx), TX Mode FSK 1.2K and FSK 2.4K Tx
	BK4819_WriteRegister(BK4819_REG_5C, 0x5665); // Enable CRC among other things we don't know yet
	BK4819_WriteRegister(BK4819_REG_5D, 0x4700); // FSK Data Length 72 Bytes (0xabcd + 2 byte length + 64 byte payload + 2 byte CRC + 0xdcba)
	
	gEnableSpeaker = false;

	BK4819_ToggleGpioOut(BK4819_GPIO0_PIN28_RX_ENABLE, false);

	BK4819_SetFilterBandwidth(gCurrentVfo->CHANNEL_BANDWIDTH);

}

// ------------------------------------------------------------------------------

void BK4819_ConfigureFSK(bool rx)
{
	init_fsk0();
	
	if(rx){
		
		BK4819_ResetFSK();
		BK4819_WriteRegister(BK4819_REG_02, 0);
		BK4819_WriteRegister(BK4819_REG_3F, 0);
		BK4819_RX_TurnOn();
		BK4819_WriteRegister(BK4819_REG_3F, 0
			| BK4819_REG_3F_FSK_RX_FINISHED
			| BK4819_REG_3F_FSK_FIFO_ALMOST_FULL
			);
	// Clear RX FIFO
	// FSK Preamble Length 7 bytes
	// FSK SyncLength Selection
		BK4819_WriteRegister(BK4819_REG_59, 0x4068);
	// Enable FSK Scramble
	// Enable FSK RX
	// FSK Preamble Length 7 bytes
	// FSK SyncLength Selection
		BK4819_WriteRegister(BK4819_REG_59, 0x3068);

//		BK4819_SetFrequency(gRxVfo->pRX->Frequency);
		BK4819_TuneTo(gRxVfo->pRX->Frequency, false);
		BK4819_ToggleGpioOut(BK4819_GPIO0_PIN28_RX_ENABLE, true);

// --------------- squelch ----------------------------------------------------------		
//	BK4819_SetupSquelch(
//		gRxVfo->SquelchOpenRSSIThresh, gRxVfo->SquelchCloseRSSIThresh,
//		gRxVfo->SquelchOpenNoiseThresh, gRxVfo->SquelchCloseNoiseThresh,
//		gRxVfo->SquelchCloseGlitchThresh, gRxVfo->SquelchOpenGlitchThresh);	
	
	}else{

		BK4819_PrepareTransmit();
		SYSTEM_DelayMs(10);

		BK4819_SelectFilter(gCurrentVfo->pTX->Frequency);
		BK4819_ToggleGpioOut(BK4819_GPIO1_PIN29_PA_ENABLE, true);
		SYSTEM_DelayMs(5);

		BK4819_SetupPowerAmplifier(gCurrentVfo->TXP_CalculatedSetting, gCurrentVfo->pTX->Frequency);
		SYSTEM_DelayMs(10);

//		BK4819_SetFrequency(gCurrentVfo->pTX->Frequency);
	
//		switch (gCurrentVfo->pTX->CodeType) {
//		case CODE_TYPE_CONTINUOUS_TONE:
//			BK4819_SetCTCSSFrequency(CTCSS_Options[gCurrentVfo->pTX->Code]);
//			break;
//		case CODE_TYPE_DIGITAL:
//		case CODE_TYPE_REVERSE_DIGITAL:
//			BK4819_SetCDCSSCodeWord(
//				DCS_GetGolayCodeWord(
//					gCurrentVfo->pTX->CodeType,
//					gCurrentVfo->pTX->Code
//					)
//				);
//			break;
//		default:
//			BK4819_ExitSubAu();
//			break;
//		}
	}


}

/*void BK4819_PlayRogerMDC(void) {
  uint8_t i;

//  BK4819_SetAF(BK4819_AF_MUTE);
  BK4819_WriteRegister(
      BK4819_REG_58,
      0x37C3); // FSK Enable, RX Bandwidth FFSK1200/1800, 0xAA or 0x55 Preamble,
               // 11 RX Gain, 101 RX Mode, FFSK1200/1800 TX
  BK4819_WriteRegister(BK4819_REG_72, 0x3065); // Set Tone2 to 1200Hz
  BK4819_WriteRegister(BK4819_REG_70,
                       0x00E0); // Enable Tone2 and Set Tone2 Gain
  BK4819_WriteRegister(BK4819_REG_5D,
                       0x0D00); // Set FSK data length to 13 bytes
  BK4819_WriteRegister(
      BK4819_REG_59,
      0x8068); // 4 byte sync length, 6 byte preamble, clear TX FIFO
  BK4819_WriteRegister(
      BK4819_REG_59,
      0x0068); // Same, but clear TX FIFO is now unset (clearing done)
  BK4819_WriteRegister(BK4819_REG_5A, 0x5555); // First two sync bytes
  BK4819_WriteRegister(BK4819_REG_5B,
                       0x55AA); // End of sync bytes. Total 4 bytes: 555555aa
  BK4819_WriteRegister(BK4819_REG_5C, 0xAA30); // Disable CRC
  for (i = 0; i < 7; i++) {
    BK4819_WriteRegister(
        BK4819_REG_5F, FSK_RogerTable[i]); // Send the data from the roger table
  }
  SYSTEM_DelayMs(20);
  BK4819_WriteRegister(BK4819_REG_59,
                       0x0868); // 4 sync bytes, 6 byte preamble, Enable FSK TX
  SYSTEM_DelayMs(180);
  // Stop FSK TX, reset Tone2, disable FSK.
  BK4819_WriteRegister(BK4819_REG_59, 0x0068);
  BK4819_WriteRegister(BK4819_REG_70, 0x0000);
  BK4819_WriteRegister(BK4819_REG_58, 0x0000);
}*/

void BK4819_Enable_AfDac_DiscMode_TxDsp(void) {
  BK4819_WriteRegister(BK4819_REG_30, 0x0000);
  BK4819_WriteRegister(BK4819_REG_30, 0x0302);
}

void BK4819_GetVoxAmp(uint16_t *pResult) {
  *pResult = BK4819_ReadRegister(BK4819_REG_64) & 0x7FFF;
}

void BK4819_PlayDTMFEx(bool bLocalLoopback, char Code) {
  BK4819_EnableDTMF();
  BK4819_EnterTxMute();
  BK4819_SetAF(bLocalLoopback ? BK4819_AF_BEEP : BK4819_AF_MUTE);
  BK4819_WriteRegister(BK4819_REG_70, 0xD3D3);
  BK4819_EnableTXLink();
  SYSTEM_DelayMs(50);
  BK4819_PlayDTMF(Code);
  BK4819_ExitTxMute();
}

void BK4819_ToggleAFBit(bool on) {
  uint16_t reg = BK4819_ReadRegister(BK4819_REG_47);
  reg &= ~(1 << 8);
  if (on)
    reg |= on << 8;
  BK4819_WriteRegister(BK4819_REG_47, reg);
}

void BK4819_ToggleAFDAC(bool on) {
  uint32_t Reg = BK4819_ReadRegister(BK4819_REG_30);
  Reg &= ~(1 << 9);
  if (on)
    Reg |= (1 << 9);
  BK4819_WriteRegister(BK4819_REG_30, Reg);
}

/*OLD CODE
void BK4819_TuneTo(uint32_t f, bool precise) {
  BK4819_SelectFilter(f);
  BK4819_SetFrequency(f);
  uint16_t reg = BK4819_ReadRegister(BK4819_REG_30);
  if (precise) {
    // BK4819_WriteRegister(BK4819_REG_30, 0);
    BK4819_WriteRegister(BK4819_REG_30, 0x0200); // from radtel-rt-890-oefw
  } else {
    BK4819_WriteRegister(BK4819_REG_30, reg & ~BK4819_REG_30_ENABLE_VCO_CALIB);
  }
  BK4819_WriteRegister(BK4819_REG_30, reg);
}*/

void BK4819_TuneTo(uint32_t f, bool precise) {//RX Only!
  BK4819_SelectFilter(f);
  BK4819_SetFrequency(f);
  uint16_t reg;
  const uint16_t regrx = 0xBFF1;  //rx
  
  if (precise) {
    reg = 0x0200; // from radtel-rt-890-oefw
  } else {
    reg = regrx & ~BK4819_REG_30_ENABLE_VCO_CALIB;
  }
  BK4819_WriteRegister(BK4819_REG_30, reg);//reset
  BK4819_WriteRegister(BK4819_REG_30, regrx);//rx
}

void BK4819_SetToneFrequency(uint16_t f) {
  BK4819_WriteRegister(0x71, (f * 103U) / 10U);
}
