import serial
import requests
import time
import threading
import xml.etree.ElementTree as ET
import re
import unicodedata
from transliterate import translit
from datetime import datetime

# Настройки порта
SERIAL_PORT = 'COM14'
BAUDRATE = 38400
TIMEOUT = 1

try:
    ser = serial.Serial(SERIAL_PORT, baudrate=BAUDRATE, timeout=TIMEOUT)
except Exception:
    ser = None

# Константы
FEEDS_FILE = 'feeds.txt'
USER_AGENT = {'User-Agent': 'Mozilla/5.0'}

# Глобальные переменные
current_task_lock = threading.Lock()
task_running = False

# ---------------- feeds.txt парсер ----------------
def parse_feeds_file(filename=FEEDS_FILE):
    header1 = ""
    header2 = ""
    header3 = ""
    cat1 = []
    cat2 = []
    cat3 = []
    section = None
    try:
        with open(filename, 'r', encoding='utf-8') as f:
            lines = [ln.rstrip('\n') for ln in f]
    except Exception:
        return header1, header2, header3, cat1, cat2, cat3

    idx = 0
    while idx < len(lines) and lines[idx].strip() == "":
        idx += 1
    if idx < len(lines):
        header1 = lines[idx]
        idx += 1
    if idx < len(lines):
        header2 = lines[idx]
        idx += 1
    if idx < len(lines):
        header3 = lines[idx]
        idx += 1

    while idx < len(lines):
        ln = lines[idx].strip()
        if ln.lower() == '[cat1]':
            section = 1
        elif ln.lower() == '[cat2]':
            section = 2
        elif ln.lower() == '[cat3]':
            section = 3
        else:
            if section == 1 and ln:
                cat1.append(ln)
            elif section == 2 and ln:
                cat2.append(ln)
            elif section == 3 and ln:
                cat3.append(ln)
        idx += 1

    return header1, header2, header3, cat1, cat2, cat3

# ---------------- нормализация и разбиение ----------------
def normalize_text(s):
    if not s:
        return ''
    s = unicodedata.normalize('NFKC', s)
    s = s.replace('\r', ' ').replace('\n', ' ').replace('\t', ' ')
    s = s.replace('\u00A0', ' ')
    s = re.sub(r'<[^>]+>', '', s)
    s = re.sub(r'http[s]?://\S+', '', s)
    s = ''.join(ch if ord(ch) >= 32 else ' ' for ch in s)
    s = re.sub(r'\s+', ' ', s).strip()
    s = s.replace('\u0404', 'E')
    return s

def chunks_by_words(s, limit=60):
    if not s:
        yield ''
        return
    words = s.split(' ')
    cur = ''
    for w in words:
        if cur == '':
            if len(w) <= limit:
                cur = w
            else:
                for i in range(0, len(w), limit):
                    yield w[i:i+limit]
                cur = ''
        else:
            if len(cur) + 1 + len(w) <= limit:
                cur = cur + ' ' + w
            else:
                yield cur
                if len(w) <= limit:
                    cur = w
                else:
                    for i in range(0, len(w), limit):
                        yield w[i:i+limit]
                    cur = ''
    if cur:
        yield cur

def translit_safe(s):
    try:
        if any('а' <= c <= 'я' or 'А' <= c <= 'Я' for c in s):
            return translit(s, 'ru', reversed=True)
        return s
    except Exception:
        return re.sub(r'[^\x00-\x7F]+', '', s)

# ---------------- RSS ----------------
def clean_text(text):
    return normalize_text(text or '')

def fetch_three_titles(url):
    try:
        resp = requests.get(url if url.startswith('http') else 'http://' + url, headers=USER_AGENT, timeout=10)
        root = ET.fromstring(resp.content)
        items = root.findall('.//item')
        titles = []
        for it in items[:3]:
            t_el = it.find('title')
            t = t_el.text if t_el is not None and t_el.text else ""
            titles.append(clean_text(t))
        if not titles:
            return "no data"
        return ' ; '.join([t for t in titles if t])
    except Exception:
        return "error"

def fetch_first_title_and_body(url):
    try:
        resp = requests.get(url if url.startswith('http') else 'http://' + url, headers=USER_AGENT, timeout=10)
        root = ET.fromstring(resp.content)
        item = root.find('.//item')
        if item is None:
            return "no data"
        title_el = item.find('title')
        title = title_el.text if title_el is not None and title_el.text else ""
        desc_el = item.find('description')
        if desc_el is None:
            desc_el = item.find('{http://purl.org/rss/1.0/modules/content/}encoded')
        desc = desc_el.text if desc_el is not None and desc_el.text else ""
        combined = f"{title} {desc}".strip()
        return clean_text(combined) or "no data"
    except Exception:
        return "error"

# ---------------- отправка в порт ----------------
def send_raw_payload_bytes(payload_bytes):
    if not payload_bytes or ser is None:
        return
    try:
        ser.write(payload_bytes)
        ser.flush()
    except Exception:
        pass
    time.sleep(0.12)

def send_packeted_text(msg_text):
    if msg_text is None:
        return

    s = normalize_text(msg_text)
    for logical in chunks_by_words(s, limit=60):
        with current_task_lock:
            # проверка выполняется извне — здесь держим лок для атомарности, но ничего не прерываем
            pass

        logical_trans = translit_safe(logical)

        if len(logical_trans) < 60:
            logical_trans = logical_trans.ljust(60)

        try:
            data = logical_trans.encode('windows-1251', errors='ignore')
        except Exception:
            data = b''

        if len(data) < 60:
            data += b' ' * (60 - len(data))

        send_raw_payload_bytes(data)

        if logical.strip() and len(logical.strip()) < 60:
            send_raw_payload_bytes(b'\r\n')

        for _ in range(15):
            with current_task_lock:
                if not task_running:
                    return
            time.sleep(1)

# ---------------- Логика команд ----------------
def send_line_n_from_feeds(n):
    try:
        with open(FEEDS_FILE, 'r', encoding='utf-8') as f:
            all_lines = [ln.rstrip('\n') for ln in f]
    except Exception:
        all_lines = []

    if 1 <= n <= len(all_lines):
        send_packeted_text(all_lines[n-1])
    else:
        send_packeted_text("no data")

def send_time_and_first_line():
    header1, header2, header3, cat1, cat2, cat3 = parse_feeds_file(FEEDS_FILE)
    now = datetime.now()
    time_part = now.strftime("--%H:%M----%d-%m-%y-")
    suffix = (header1 or "")[:40].ljust(40, ' ')
    first_message = f"{time_part}{suffix}"
    send_packeted_text(first_message)

def send_lines_2_3_4_with_delays():
    try:
        with open(FEEDS_FILE, 'r', encoding='utf-8') as f:
            lines = [ln.rstrip('\n') for ln in f]
    except Exception:
        lines = []

    for n in (2, 3, 4):
        if n <= len(lines):
            send_packeted_text(lines[n-1])
        else:
            send_packeted_text("no data")
        for _ in range(15):
            with current_task_lock:
                if not task_running:
                    return
            time.sleep(1)

def send_cat1_rss_with_delays():
    _, _, _, cat1, _, _ = parse_feeds_file(FEEDS_FILE)
    if not cat1:
        send_packeted_text("no data")
        return
    for url in cat1:
        send_packeted_text(fetch_three_titles(url))
        for _ in range(15):
            with current_task_lock:
                if not task_running:
                    return
            time.sleep(1)

def send_cat2_rss_with_delays():
    _, _, _, _, cat2, _ = parse_feeds_file(FEEDS_FILE)
    if not cat2:
        send_packeted_text("no data")
        return
    for url in cat2:
        send_packeted_text(fetch_first_title_and_body(url))
        for _ in range(15):
            with current_task_lock:
                if not task_running:
                    return
            time.sleep(1)

def send_cat3_rss_with_delays():
    # Аналогично cat2
    _, _, _, _, _, cat3 = parse_feeds_file(FEEDS_FILE)
    if not cat3:
        send_packeted_text("no data")
        return
    for url in cat3:
        send_packeted_text(fetch_first_title_and_body(url))
        for _ in range(15):
            with current_task_lock:
                if not task_running:
                    return
            time.sleep(1)

# ---------------- слушатель команд ----------------
def extract_command_digit_from_message(raw_bytes):
    try:
        s = raw_bytes.decode('windows-1251', errors='ignore')
    except Exception:
        try:
            s = raw_bytes.decode(errors='ignore')
        except Exception:
            return None
    if len(s) >= 14:
        ch = s[13]
        if ch.isdigit():
            return ch
    return None

def command_worker(digit):
    global task_running
    try:
        if digit == '0':
            send_line_n_from_feeds(5)
        elif digit == '1':
            send_time_and_first_line()
        elif digit == '2':
            send_lines_2_3_4_with_delays()
        elif digit == '3':
            send_cat1_rss_with_delays()
        elif digit == '4':
            send_cat2_rss_with_delays()
        elif digit == '5':
            send_cat3_rss_with_delays()
        else:
            send_packeted_text("no action for this digit")
    finally:
        with current_task_lock:
            task_running = False

def listen_for_commands():
    global task_running
    while True:
        if ser is None:
            time.sleep(1)
            continue
        try:
            raw = ser.read(200)
            if not raw:
                continue
            digit = extract_command_digit_from_message(raw)
            if digit is None:
                continue
            with current_task_lock:
                if task_running:
                    continue
                else:
                    task_running = True
            thr = threading.Thread(target=command_worker, args=(digit,), daemon=True)
            thr.start()
        except Exception:
            continue

# ---------------- Запуск ----------------
if ser:
    listener_thread = threading.Thread(target=listen_for_commands, daemon=True)
    listener_thread.start()
    try:
        while True:
            time.sleep(1)
    except KeyboardInterrupt:
        pass
else:
    print("COM порт не инициализирован")
